(function () {
    const formatNumber = (n) => {
      const num = typeof n === 'number' ? n : Number(String(n).replace(/,/g, ''));
      if (Number.isNaN(num)) return String(n);
      return num.toLocaleString('en-US');
    };
    const parsePrice = (v) => {
      if (v == null) return 0;
      return Number(String(v).replace(/[, ]+/g, '')) || 0;
    };
  
    // elements
    const cardsContainer = document.querySelector('.pm-cards');
    const listEl = document.getElementById('pm-selected-list');
    const totalEl = document.getElementById('pm-total');
    const currencySelect = document.getElementById('pm-currency');
    const discountInput = document.getElementById('pm-discount-code');
    const applyBtn = document.getElementById('pm-apply-discount');
    const alertEl = document.getElementById('pm-alert');
    const successEl = document.getElementById('pm-success');
  
    const RATE_TOMAN_TO_USD = 42000;
    const VALID_DISCOUNTS = { VIP10: 10, NEW5: 5 };
    let discount = { code: null, value: 0 };

    function getCardRows() {
      const cards = Array.from(document.querySelectorAll('.pm-card'));
      return cards.map((card, idx) => {
        let cb = card.querySelector('.pm-checkbox');
        if (!cb) {
          cb = document.createElement('input');
          cb.type = 'checkbox';
          cb.className = 'pm-checkbox';
          card.insertBefore(cb, card.firstChild);
        }
        if (!cb.id) {
          cb.id = `svc-auto-${idx + 1}`;
        }
        return { card, checkbox: cb, index: idx };
      });
    }
    function getSelectedItems() {
      const rows = getCardRows();
      return rows
        .filter(r => r.checkbox.checked)
        .map(r => {
          const cb = r.checkbox;
          return {
            index: r.index,
            id: cb.id,
            title: cb.dataset.title || cb.getAttribute('aria-label') || (r.card.querySelector('.title')?.textContent || '').trim(),
            price: parsePrice(cb.dataset.price || cb.value || r.card.querySelector('.pm-price')?.textContent || 0)
          };
        });
    }
  
    function updateSelectedList() {
      const items = getSelectedItems();
      listEl.innerHTML = '';
      if (items.length === 0) {
        const empty = document.createElement('div');
        empty.className = 'pm-empty';
        empty.textContent = 'موردی انتخاب نشده';
        listEl.appendChild(empty);
        return;
      }
  
      items.forEach(it => {
        const chip = document.createElement('div');
        chip.className = 'pm-chip';
        const removeBtn = `<button class="pm-remove" aria-label="حذف ${it.title}" data-target="${it.index}">&times;</button>`;
        chip.innerHTML = `<span>${it.title} — ${formatNumber(it.price)} تومان</span> ${removeBtn}`;
        listEl.appendChild(chip);
      });
    }

    function calculateTotal() {
      const items = getSelectedItems();
      let sum = items.reduce((s, i) => s + Number(i.price || 0), 0);
      if (discount.value) sum = Math.round(sum * (1 - discount.value / 100));
      const currency = currencySelect ? currencySelect.value : 'toman';
      if (currency === 'toman') {
        totalEl.textContent = `${formatNumber(sum)} تومان`;
      } else {
        const usd = sum / RATE_TOMAN_TO_USD;
        // show with two decimals but format thousands if >1000
        const txt = Number.isFinite(usd) ? usd.toFixed(2) : '0.00';
        // format integer part with commas
        const parts = txt.split('.');
        parts[0] = Number(parts[0]).toLocaleString('en-US');
        totalEl.textContent = `${parts.join('.')} $`;
      }
    }

    function attachCardClickHandlers() {
      if (!cardsContainer) return;
      cardsContainer.addEventListener('click', (e) => {
        const card = e.target.closest('.pm-card');
        if (!card) return;
        const interactive = e.target.closest('button, a, input, .pm-remove');
        if (interactive && !interactive.classList.contains('pm-remove')) {
          return;
        }

        const cb = card.querySelector('.pm-checkbox');
        if (!cb) return;
        if (e.target === cb) {
          return;
        }
        cb.checked = !cb.checked;
        cb.dispatchEvent(new Event('change', { bubbles: true }));
      }, false);
    }
    function attachCheckboxChangeHandlers() {
      document.addEventListener('change', (e) => {
        if (!e.target.matches('.pm-checkbox')) return;
        updateSelectedList();
        calculateTotal();
      }, false);
    }

    function attachChipRemoveHandler() {
      listEl.addEventListener('click', (e) => {
        const btn = e.target.closest('.pm-remove');
        if (!btn) return;
        const idx = Number(btn.dataset.target);
        const rows = getCardRows();
        const row = rows[idx];
        if (row && row.checkbox) {
          row.checkbox.checked = false;
          row.checkbox.dispatchEvent(new Event('change', { bubbles: true }));
        } else {
          const parentText = btn.parentElement?.querySelector('span')?.textContent || '';
          const title = parentText.split(' — ')[0]?.trim();
          if (title) {
            const rowsFind = rows.find(r => {
              const t = r.checkbox.dataset.title || r.card.querySelector('.title')?.textContent;
              return (t || '').trim() === title;
            });
            if (rowsFind) {
              rowsFind.checkbox.checked = false;
              rowsFind.checkbox.dispatchEvent(new Event('change', { bubbles: true }));
            }
          }
        }
      }, false);
    }

    function attachDiscountHandler() {
      if (!applyBtn) return;
      applyBtn.addEventListener('click', () => {
        if (!discountInput) return;
        const code = (discountInput.value || '').trim().toUpperCase();
        if (alertEl) alertEl.style.display = 'none';
        if (successEl) successEl.style.display = 'none';
        if (!code) {
          if (alertEl) {
            alertEl.textContent = 'لطفا کد تخفیف را وارد کنید.';
            alertEl.style.display = 'block';
          }
          return;
        }
        if (VALID_DISCOUNTS[code]) {
          discount.code = code;
          discount.value = VALID_DISCOUNTS[code];
          if (successEl) {
            successEl.textContent = `کد ${code} اعمال شد — ${discount.value}% تخفیف.`;
            successEl.style.display = 'block';
          }
          calculateTotal();
        } else {
          if (alertEl) {
            alertEl.textContent = 'کد تخفیف معتبر نیست.';
            alertEl.style.display = 'block';
          }
        }
      }, false);
    }

    function attachCurrencyHandler() {
      if (!currencySelect) return;
      currencySelect.addEventListener('change', () => {
        calculateTotal();
      });
    }

    function init() {
      attachCardClickHandlers();
      attachCheckboxChangeHandlers();
      attachChipRemoveHandler();
      attachDiscountHandler();
      attachPaymentHandlers();
      attachCurrencyHandler();
      updateSelectedList();
      calculateTotal();
    }
  
    if (document.readyState === 'loading') {
      document.addEventListener('DOMContentLoaded', init);
    } else {
      init();
    }
  })();
  


  /*--modal--*/
  function attachPaymentHandlers() {
    const manual = document.getElementById('pm-manual');
    const foreign = document.getElementById('pm-foreign');

    if (manual) {
      manual.addEventListener('click', () => openModal('manualModal'));
    }

    if (foreign) {
      foreign.addEventListener('click', () => openModal('foreignModal'));
    }
    function openModal(id) {
      const modal = document.getElementById(id);
      if (!modal) return;
  
      modal.style.display = 'flex';
  
      const closeBtn = modal.querySelector('.close-btn');
      closeBtn.addEventListener('click', () => {
        modal.style.display = 'none';
      });
  
      window.addEventListener('click', (e) => {
        if (e.target === modal) {
          modal.style.display = 'none';
        }
      });
    }
  }
  
  